/* nwreckdum Quake .pak manipulator for linux.*/

#include <ctype.h>
#include <dirent.h>
#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <unistd.h>

#include <stdarg.h>
#include <ctype.h>
#include <limits.h>


/* $Id: shhopt.h,v 1.2 1996/06/06 00:06:35 sverrehu Exp $ */
#ifndef SHHOPT_H
#define SHHOPT_H

/* constants for recognized option types. */
typedef enum {
    OPT_END,               /* nothing. used as ending element. */
    OPT_FLAG,              /* no argument following. sets variable to 1. */
    OPT_STRING,            /* string argument. */
    OPT_INT,               /* signed integer argument. */
    OPT_UINT,              /* unsigned integer argument. */
    OPT_LONG,              /* signed long integer argument. */
    OPT_ULONG,             /* unsigned long integer argument. */
} optArgType;

/* flags modifying the default way options are handeled. */
#define OPT_CALLFUNC  1    /* pass argument to a function. */

typedef struct {
    char       shortName;  /* Short option name. */
    char       *longName;  /* Long option name, no including '--'. */
    optArgType type;       /* Option type. */
    void       *arg;       /* Pointer to variable to fill with argument,
                            * or pointer to function if Type == OPT_FUNC. */
    int        flags;      /* Modifier flags. */
} optStruct;


void optSetFatalFunc(void (*f)(const char *, ...));
void optParseOptions(int *argc, char *argv[],
		     optStruct opt[], int allowNegNum);

#endif


/* $Id: shhopt.c,v 1.2 1996/06/06 00:06:34 sverrehu Exp $ */
/**************************************************************************
 *
 *  FILE            shhopt.c
 *
 *  DESCRIPTION     Functions for parsing command line arguments. Values
 *                  of miscellaneous types may be stored in variables,
 *                  or passed to functions as specified.
 *
 *  REQUIREMENTS    Some systems lack the ANSI C -function strtoul. If your
 *                  system is one of those, you'll ned to write one yourself,
 *                  or get the GNU liberty-library (from prep.ai.mit.edu).
 *
 *  WRITTEN BY      Sverre H. Huseby <sverrehu@ifi.uio.no>
 *
 **************************************************************************/


/**************************************************************************
 *                                                                        *
 *                       P R I V A T E    D A T A                         *
 *                                                                        *
 **************************************************************************/

static void optFatalFunc(const char *, ...);
static void (*optFatal)(const char *format, ...) = optFatalFunc;



/**************************************************************************
 *                                                                        *
 *                   P R I V A T E    F U N C T I O N S                   *
 *                                                                        *
 **************************************************************************/

/*-------------------------------------------------------------------------
 *
 *  NAME          optFatalFunc
 *
 *  FUNCTION      Show given message and abort the program.
 *
 *  INPUT         format, ...
 *                        Arguments used as with printf().
 *
 *  RETURNS       Never returns. The program is aborted.
 *
 */
void optFatalFunc(const char *format, ...)
{
    va_list ap;

    fflush(stdout);
    va_start(ap, format);
    vfprintf(stderr, format, ap);
    va_end(ap);
    exit(99);
}



/*-------------------------------------------------------------------------
 *
 *  NAME          optStructCount
 *
 *  FUNCTION      Get number of options in a optStruct.
 *
 *  INPUT         opt     array of possible options.
 *
 *  RETURNS       Number of options in the given array.
 *
 *  DESCRIPTION   Count elements in an optStruct-array. The strcture must
 *                be ended using an element of type OPT_END.
 *
 */
static int optStructCount(optStruct opt[])
{
    int ret = 0;

    while (opt[ret].type != OPT_END)
        ++ret;
    return ret;
}



/*-------------------------------------------------------------------------
 *
 *  NAME          optMatch
 *
 *  FUNCTION      Find a matching option.
 *
 *  INPUT         opt     array of possible options.
 *                s       string to match, without `-' or `--'.
 *                lng     match long option, otherwise short.
 *
 *  RETURNS       Index to the option if found, -1 if not found.
 *
 *  DESCRIPTION   Short options are matched from the first character in
 *                the given string.
 *
 */
static int optMatch(optStruct opt[], const char *s, int lng)
{
    int  nopt, q, matchlen = 0;
    char *p;

    nopt = optStructCount(opt);
    if (lng) {
	if ((p = strchr(s, '=')) != NULL)
	    matchlen = p - s;
	else
	    matchlen = strlen(s);
    }
    for (q = 0; q < nopt; q++) {
	if (lng) {
	    if (!opt[q].longName)
		continue;
	    if (strncmp(s, opt[q].longName, matchlen) == 0)
		return q;
	} else {
	    if (!opt[q].shortName)
		continue;
	    if (*s == opt[q].shortName)
		return q;
	}
    }
    return -1;
}



/*-------------------------------------------------------------------------
 *
 *  NAME          optString
 *
 *  FUNCTION      Return a (static) string with the option name.
 *
 *  INPUT         opt     the option to stringify.
 *                lng     is it a long option?
 *
 *  RETURNS       Pointer to static string.
 *
 */
static char *optString(optStruct *opt, int lng)
{
    static char ret[31];

    if (lng) {
	strcpy(ret, "--");
	strncpy(ret + 2, opt->longName, 28);
    } else {
	ret[0] = '-';
	ret[1] = opt->shortName;
	ret[2] = '\0';
    }
    return ret;
}



/*-------------------------------------------------------------------------
 *
 *  NAME          optNeedsArgument
 *
 *  FUNCTION      Check if an option requires an argument.
 *
 *  INPUT         opt     the option to check.
 *
 *  RETURNS       Boolean value.
 *
 */
static int optNeedsArgument(optStruct *opt)
{
    return opt->type == OPT_STRING
	|| opt->type == OPT_INT
	|| opt->type == OPT_UINT
	|| opt->type == OPT_LONG
	|| opt->type == OPT_ULONG;
}



/*-------------------------------------------------------------------------
 *
 *  NAME          argvRemove
 *
 *  FUNCTION      Remove an entry from an argv-array.
 *
 *  INPUT         argc    pointer to number of options.
 *                argv    array of option-/argument-strings.
 *                i       index of option to remove.
 *
 *  OUTPUT        argc    new argument count.
 *                argv    array with given argument removed.
 *
 */
static void argvRemove(int *argc, char *argv[], int i)
{
    if (i >= *argc)
        return;
    while (i++ < *argc)
        argv[i - 1] = argv[i];
    --*argc;
}



/*-------------------------------------------------------------------------
 *
 *  NAME          optExecute
 *
 *  FUNCTION      Perform the action of an option.
 *
 *  INPUT         opt     array of possible options.
 *                arg     argument to option, if it applies.
 *                lng     was the option given as a long option?
 *
 *  RETURNS       Nothing. Aborts in case of error.
 *
 */
void optExecute(optStruct *opt, char *arg, int lng)
{
    switch (opt->type) {
      case OPT_FLAG:
	if (opt->flags & OPT_CALLFUNC)
	    ((void (*)(void)) opt->arg)();
	else
	    *((int *) opt->arg) = 1;
	break;

      case OPT_STRING:
	if (opt->flags & OPT_CALLFUNC)
	    ((void (*)(char *)) opt->arg)(arg);
	else
	    *((char **) opt->arg) = arg;
	break;

      case OPT_INT:
      case OPT_LONG: {
	  long tmp;
	  char *e;

	  tmp = strtol(arg, &e, 10);
	  if (*e)
	      optFatal("invalid number `%s'\n", arg);
	  if (errno == ERANGE
	      || (opt->type == OPT_INT && (tmp > INT_MAX || tmp < INT_MIN)))
	      optFatal("number `%s' to `%s' out of range\n",
		       arg, optString(opt, lng));
	  if (opt->type == OPT_INT) {
	      if (opt->flags & OPT_CALLFUNC)
		  ((void (*)(int)) opt->arg)((int) tmp);
	      else
		  *((int *) opt->arg) = (int) tmp;
	  } else /* OPT_LONG */ {
	      if (opt->flags & OPT_CALLFUNC)
		  ((void (*)(long)) opt->arg)(tmp);
	      else
		  *((long *) opt->arg) = tmp;
	  }
	  break;
      }

      case OPT_UINT:
      case OPT_ULONG: {
	  unsigned long tmp;
	  char *e;

	  tmp = strtoul(arg, &e, 10);
	  if (*e)
	      optFatal("invalid number `%s'\n", arg);
	  if (errno == ERANGE
	      || (opt->type == OPT_UINT && tmp > UINT_MAX))
	      optFatal("number `%s' to `%s' out of range\n",
		       arg, optString(opt, lng));
	  if (opt->type == OPT_UINT) {
	      if (opt->flags & OPT_CALLFUNC)
		  ((void (*)(unsigned)) opt->arg)((unsigned) tmp);
	      else
		  *((unsigned *) opt->arg) = (unsigned) tmp;
	  } else /* OPT_ULONG */ {
	      if (opt->flags & OPT_CALLFUNC)
		  ((void (*)(unsigned long)) opt->arg)(tmp);
	      else
		  *((unsigned long *) opt->arg) = tmp;
	  }
	  break;
      }

      default:
	break;
    }
}



/**************************************************************************
 *                                                                        *
 *                    P U B L I C    F U N C T I O N S                    *
 *                                                                        *
 **************************************************************************/

/*-------------------------------------------------------------------------
 *
 *  NAME          optSetFatalFunc
 *
 *  FUNCTION      Set function used to display error message and exit.
 *
 *  SYNOPSIS      #include "shhmsg.h"
 *                void optSetFatalFunc(void (*f)(const char *, ...));
 *
 *  INPUT         f       function accepting printf()'like parameters,
 *                        that _must_ abort the program.
 *
 */
void optSetFatalFunc(void (*f)(const char *, ...))
{
    optFatal = f;
}



/*-------------------------------------------------------------------------
 *
 *  NAME          optParseOptions
 *
 *  FUNCTION      Parse commandline options.
 *
 *  SYNOPSIS      #include "shhopt.h"
 *                void optParseOptions(int *argc, char *argv[],
 *                                     optStruct opt[], int allowNegNum);
 *
 *  INPUT         argc    Pointer to number of options.
 *                argv    Array of option-/argument-strings.
 *                opt     Array of possible options.
 *                allowNegNum
 *                        a negative number is not to be taken as
 *                        an option.
 *
 *  OUTPUT        argc    new argument count.
 *                argv    array with arguments removed.
 *
 *  RETURNS       Nothing. Aborts in case of error.
 *
 *  DESCRIPTION   This function checks each option in the argv-array
 *                against strings in the opt-array, and `executes' any
 *                matching action. Any arguments to the options are
 *                extracted and stored in the variables or passed to
 *                functions pointed to by entries in opt.
 *
 *                Options and arguments used are removed from the argv-
 *                array, and argc is decreased accordingly.
 *
 *                Any error leads to program abortion.
 *
 */
void optParseOptions(int *argc, char *argv[], optStruct opt[], int allowNegNum)
{
    int  ai,        /* argv index. */
         optarg,    /* argv index of option argument, or -1 if none. */
         mi,        /* Match index in opt. */
         done;
    char *arg,      /* Pointer to argument to an option. */
         *o,        /* pointer to an option character */
         *p;

    /*
     *  Loop through all arguments.
     */
    for (ai = 0; ai < *argc; ) {
	/*
	 *  "--" indicates that the rest of the argv-array does not
	 *  contain options.
	 */
	if (strcmp(argv[ai], "--") == 0) {
            argvRemove(argc, argv, ai);
	    break;
	}

	if (allowNegNum && argv[ai][0] == '-' && isdigit(argv[ai][1])) {
	    ++ai;
	    continue;
	} else if (strncmp(argv[ai], "--", 2) == 0) {
	    /* long option */
	    /* find matching option */
	    if ((mi = optMatch(opt, argv[ai] + 2, 1)) < 0)
		optFatal("unrecognized option `%s'\n", argv[ai]);

	    /* possibly locate the argument to this option. */
	    arg = NULL;
	    if ((p = strchr(argv[ai], '=')) != NULL)
		arg = p + 1;

	    /* does this option take an argument? */
	    optarg = -1;
	    if (optNeedsArgument(&opt[mi])) {
		/* option needs an argument. find it. */
		if (!arg) {
		    if ((optarg = ai + 1) == *argc)
			optFatal("option `%s' requires an argument\n",
				 optString(&opt[mi], 1));
		    arg = argv[optarg];
		}
	    } else {
		if (arg)
		    optFatal("option `%s' doesn't allow an argument\n",
			     optString(&opt[mi], 1));
	    }
	    /* perform the action of this option. */
	    optExecute(&opt[mi], arg, 1);
	    /* remove option and any argument from the argv-array. */
            if (optarg >= 0)
                argvRemove(argc, argv, ai);
            argvRemove(argc, argv, ai);
	} else if (*argv[ai] == '-') {
	    /* A dash by itself is not considered an option. */
	    if (argv[ai][1] == '\0') {
		++ai;
		continue;
	    }
	    /* Short option(s) following */
	    o = argv[ai] + 1;
	    done = 0;
	    optarg = -1;
	    while (*o && !done) {
		/* find matching option */
		if ((mi = optMatch(opt, o, 0)) < 0)
		    optFatal("unrecognized option `-%c'\n", *o);

		/* does this option take an argument? */
		optarg = -1;
		arg = NULL;
		if (optNeedsArgument(&opt[mi])) {
		    /* option needs an argument. find it. */
		    arg = o + 1;
		    if (!*arg) {
			if ((optarg = ai + 1) == *argc)
			    optFatal("option `%s' requires an argument\n",
				     optString(&opt[mi], 0));
			arg = argv[optarg];
		    }
		    done = 1;
		}
		/* perform the action of this option. */
		optExecute(&opt[mi], arg, 0);
		++o;
	    }
	    /* remove option and any argument from the argv-array. */
            if (optarg >= 0)
                argvRemove(argc, argv, ai);
            argvRemove(argc, argv, ai);
	} else {
	    /* a non-option argument */
	    ++ai;
	}
    }
}


/* nwreckdum Quake .pak manipulator for linux.*/
#define kBufferSize    10000

#ifdef DEBUG
# define D(x)	(x)
#else
# define D(x)
#endif

typedef struct {
	char	name[56];
	unsigned int	offset, length;
} dirrec;

int	asmpak(FILE *);
int	extrpak(FILE *, int);
int	write_rec(dirrec, FILE *);
int	makepath(char *);

void	usage(void) {
	printf(
"Usage: nwreckdum [argument ...]\n"
"\n"
"  -c, --create		create PACK instead of extracting\n"
"  -d, --dir=DIR	use DIR as the input/output directory\n"
"  -l, --list		list files stored in PACK\n"
"  -h, --help		display this help\n"
"  -p, --pak=FILE	read FILE for PACK information\n"
"\n"
"Copyright 1998 by n, aka Daniel Reed.\n"
	);
	exit(0);
}

int	main(int argc, char **argv) {
	FILE	*pakhand = NULL;
	int	makepak = 0, listpak = 0;
	char	*pakfile = NULL, *outdir = NULL;
	const char	defpakfile[] = "pak0.pak",
			defoutdir[] = "pak0";
	optStruct opt[] = {
		{ 'p', "pak",	OPT_STRING,	&pakfile,	0 },
		{ 'd', "dir",	OPT_STRING,	&outdir,	0 },
		{ 'c', "create", OPT_FLAG,	&makepak,	0 },
		{ 'l', "list",	OPT_FLAG,	&listpak,	0 },
		{ 'h', "help",	OPT_FLAG,	usage,	OPT_CALLFUNC },
		{ 0,   NULL,	OPT_END,	0,		0 }
	};

	optParseOptions(&argc, argv, opt, 0);
	if (pakfile == NULL)
		pakfile = (char *)defpakfile;
	if (outdir == NULL)
		outdir = (char *)defoutdir;

	printf("Using pak file: %s\n", pakfile);
	if (!listpak)
		printf("Input/output directory: %s\n", outdir);
	printf("Operation: %s\n", (makepak?"create":listpak?"list":"extract"));

	if (makepak)
		pakhand = fopen(pakfile, "wb");
	else
		pakhand = fopen(pakfile, "rb");
	if (pakhand == NULL) {
		printf("Error opening %s: %s\n", pakfile,
			strerror(errno));
		return(-1);
	}

	if (!listpak) {

	if (!makepak && (mkdir(outdir, 493 /*octal=755*/) == -1)) {
		printf("Error making %s: %s\n", outdir, strerror(errno));
		fclose(pakhand);
		return(-1);
	}
	if (chdir(outdir) == -1) {
		printf("Error changing directories to %s: %s\n", outdir,
			strerror(errno));
		fclose(pakhand);
		return(-1);
	} }

	if (makepak)
		asmpak(pakhand);
	else
		extrpak(pakhand, listpak);

	fclose(pakhand);
	return(0);
}

int	handledir(char *base, dirrec **dir, unsigned long *filecnt) {
	DIR	*workdir = NULL;
	struct dirent	*direntry = NULL;
	struct stat	buf;

	if ((workdir = opendir(".")) == NULL) {
		printf("Unable to open work directory: %s\n",
			strerror(errno));
		return(-1);
	}
	while ((direntry = readdir(workdir)) != NULL) {
		char	basebuf[56];

		if (direntry->d_name[0] == '.')
			continue;
		if (stat(direntry->d_name, &buf) == -1)
			buf.st_mode = 0;

		if ((base != NULL) && (*base != 0))
#ifndef DOS
			snprintf(basebuf, 56, "%s/%s", base,
				direntry->d_name);
#else
		{
			strncpy(basebuf, base, 55);
			strncat(basebuf, "/", 55-strlen(basebuf));
			strncat(basebuf, direntry->d_name,
				55-strlen(basebuf));
		}
#endif /* DOS */
		else {
			strncpy(basebuf, direntry->d_name, 55);
			basebuf[55] = 0;
		}

		if (S_ISDIR(buf.st_mode)) {
			chdir(direntry->d_name);
			handledir(basebuf, dir, filecnt);
			chdir("..");
		} else {
			(*filecnt)++;
			if (*dir == NULL)
				*dir = malloc(sizeof(dirrec));
			else
				*dir = realloc(*dir, sizeof(dirrec) *
					(*filecnt));
			if (*dir == NULL) {
				printf("Memory failure: %s\n",
					strerror(errno));
				return(-1);
			}
#define CURDIRENT	((*dir)[(*filecnt)-1])
			strncpy(CURDIRENT.name, basebuf, 55);
			(CURDIRENT.name)[55] = CURDIRENT.offset =
				CURDIRENT.length = 0;
		}
	}
	closedir(workdir);
	return(1);
}

int	asmpak(FILE *out) {
	unsigned long	filecnt = 0, i = 0;
	dirrec	*dir = NULL;
	FILE	*curfile = NULL;

	fseek(out, 0, SEEK_SET);
	fwrite("PACK", 4, 1, out);
	handledir(NULL, &dir, &filecnt);
	fseek(out, 12, SEEK_SET);
	printf("Num Size     File name\n");
	printf("--- -------- -------------------------------------------------------\n");
	for (i = 0; i < filecnt; i++) {
		char	inc = 0;

		if ((curfile = fopen(dir[i].name, "r")) == NULL) {
			printf("Error opening %s: %s\n", dir[i].name,
				strerror(errno));
			exit(-1);
		}
		dir[i].offset = ftell(out);
		fseek(curfile, 0, SEEK_END);
		dir[i].length = ftell(curfile);
		printf("%3lu %8u %s\n", i, dir[i].length,
			dir[i].name);
		fseek(curfile, 0, SEEK_SET);
		while (!feof(curfile)) {
			fread(&inc, 1, 1, curfile);
			fwrite(&inc, 1, 1, out);
		}
		fclose(curfile);
	}
	{
		int	cnt = 0;
		unsigned long	offtmp = ftell(out);

		D(printf("offtmp = %lu\n", offtmp));
		for (cnt = 0; cnt < (int)filecnt; cnt++)
			fwrite(&(dir[cnt]), sizeof(dirrec), 1, out);
		fseek(out, 4, SEEK_SET);
		fwrite(&offtmp, 4, 1, out);
		offtmp = filecnt*sizeof(dirrec);
		D(printf("offtmp = %lu\n", offtmp));
		fwrite(&offtmp, 4, 1, out);
	}
	free(dir);
	return(1);
}

int	extrpak(FILE *in, int listpak) {
	dirrec	*dir = NULL;
	char	signature[4];
	unsigned long	dir_start = 0, num_ents = 0;
	int	i = 0;

	fread((void *)signature, 4, 1, in);
	if (strncmp(signature, "PACK", 4) != 0) {
		fprintf(stderr, "Error: not a PAK file\n");
		fclose(in);
		return(-1);
	}

	fread((void *)&dir_start, 4, 1, in);
	D(printf("dir_start: %lu\n", dir_start));
	fread((void *)&num_ents, 4, 1, in);
	D(printf("num_ents: %lu\n", num_ents));
	num_ents /= sizeof(dirrec);
	D(printf("num_ents: %lu\n", num_ents));
	fseek(in, dir_start, SEEK_SET);
	D(printf("ftell: %lu\n", ftell(in)));

	if ((dir = calloc(num_ents, sizeof(dirrec))) == NULL) {
		printf("Error allocating memory: %s\n", strerror(errno));
		fclose(in);
		exit(-1);
	}

	fread((void *)dir, sizeof(dirrec), num_ents, in);

	printf("Num Size     File name\n");
	printf("--- -------- -------------------------------------------------------\n");
	for (i = 0; i < (int)num_ents; i++) {
		printf("%3i %8u %s\n", i, dir[i].length,
			dir[i].name);
		if (listpak)
			continue;
		if (write_rec(dir[i], in))
			printf("Error writing record: %s\n",
				strerror(errno));
	}

	free(dir);
	return(0);
}

int	write_rec(dirrec dir, FILE *in) {
	char	buf[kBufferSize];
	FILE	*f = NULL;
	long	l = 0;
	int	blocks = 0, count = 0, s = 0;

	makepath(dir.name);
	if ((f = fopen(dir.name,"wb")) == NULL) {
		printf("Error opening %s: %s\n", dir.name,
			strerror(errno));
		return(-1);
	}
	fseek(in, dir.offset, SEEK_SET);
	blocks = dir.length / (kBufferSize-1);
	if (dir.length%(kBufferSize-1))
		blocks++;
	for (count = 0; count < blocks; count++) {
		fread((void *)buf, 1, (kBufferSize-1), in);
		if ((l = dir.length - (long)count * (kBufferSize-1)) >
			(kBufferSize-1))
			l = (kBufferSize-1);
		s = l;
		fwrite((void *)buf, 1, s, f);
	}
	fclose(f);
	return(0);
}

int	makepath(char *fname) {
/* given "some/path/to/file", makes all dirs (ie "some/", "some/path/", */
/* and "some/path/to/", so that file can be created                     */
	char	*c = fname,
		*o = (char *)malloc(strlen(fname)+1),
		*n = o;

	while(*c) {
		while(*c && (*c != '/'))
			*n++ = *c++;
		if(*c) {
			*n = '\0';
			mkdir(o, 493 /*octal=755*/);
			*n = '/';
			n++;
			c++;
		}
	}
	free(o);
	return(0);
}
